########## Filter data ###########
# This function takes cleaned data from 1_Clean raw and will:
# - Load data from Clean raw
# - Extract Perievent time course for each event for each unit:
# 	- time window for extraction. Window defined in config as: "before" and "after"
#	- if needed (config: "normalize.spikes.per.event") will normalize the mean perievent firing to mean baseline firing (baseline window defined config: "min.time.normalization" and "max.time.normalization")
# - Threshold each unit for up/down firing (complex!):
#	- if "one.sample.t.threshold" set to FALSE will separatly compare each time bin defined in "threshold.min.time" and "threshold.max.time" to "z.threshold".
#		- if at time point 0 or any time bin defined as above the threshold is passed will set the Unit firing Direction to Up/Down accordingly.
#			NOTE: This will create ONE thresholded value for a given Unit.
#	- if "one.sample.t.threshold" set to TRUE will do one.sample t.test. Way depends on setting "one.sample.to.zero":
#		- if one.sample.to.zero = TRUE, will set the mu = 0, and will run one.sample t.test for all time.bins defined in "threshold.min.time" and "threshold.max.time".
#			NOTE: that this will create basically a time vector for Direction. 
#			For given unit the time bins defined in the window will be thresholded as Up/Down/Non-responded separatly, and not as single entity!
#		- if one.sample.to.zero = FALSE, will calculate the mean firing rate across ALL occuring Events (1 to X) for given EventID (i.e. Precue Poke) and:
#			- use this value as mu to run series of one sample t.tests independently for each time bin defined in the threshold window (as above) and then threshold the unit accoridingly
#			NOTE: This will create ONE thresholded value for a given Unit.
# - Saves the extracted Perievents for each Unit, indices for each EventID/Event number and thresholded perivent as separate data frame.

# Main function ---------------------------------------------------------------------------------------------------
filter_data <- function(config.file = './R/config.R'){
  tryCatch({
    
    # Load libraries
    suppressWarnings(library(tidyverse))
    suppressWarnings(library(futile.logger))
    
    suppressWarnings(library(furrr))
    plan(multiprocess) #multiprocess
    #options(future.globals.maxSize = 891289600)
    
    # Load config file
    source(config.file)
    source('./R/2_filter_data_functions.R')
    
    # Set up logger
    flog.logger(name = log.name, threshold = threshold, appender = appender.tee(paste0(output.folder, log.name, ".log")))
    flog.threshold(threshold, name = log.name)
    
    flog.info(paste0("Initializing: ", format(Sys.Date(), "%d %B, %Y")), name = log.name)
    
    #load data
    flog.info("Loading data", name = log.name)
    Data <- read_rds(path = paste0(output.folder, cleaned.raw.data))
    
    # Get perievents
    flog.info("Init perievent", name = log.name)
    flog.info(paste0("Permute: ", permute), name = log.name)
    Data.perievents <- get_perievents(Data, 
                           group.columns = group.columns, 
                           protocol.columns = protocol.columns, 
                           perievent.columns = perievent.columns,
                           before = before, after = after,
                           bin.size = bin.size, permute = FALSE)
    
    flog.info("Saving perievent data...", name = log.name)
    write_rds(Data.perievents$Idx, path = paste0(output.folder, Idx.data.name), "gz", compression = 9L)
    write_rds(Data.perievents$Data.event, path = paste0(output.folder, Perievent.data), "gz", compression = 9L)
    
    # Normalize and threshold data
    flog.info("Init threshold", name = log.name)
    Data.threshold <- threshold_units(Data.perievents$Data.event, group.columns, 
                                      normalize.spikes.per.event, min.time.normalization, max.time.normalization,
                                      z.threshold, threshold.min.time, threshold.max.time, 
                                      one.sample.t.threshold, one.sample.to.zero)
    
    flog.info("Saving threshold data...", name = log.name)
    write_rds(Data.threshold, path = paste0(output.folder, Units.thresholded), "gz", compression = 9L)
    
    flog.info("It is done!", name = log.name)
    
	  rm(list = ls())
	  gc()
	  
    return(NULL)
    
  }, error = function(e) {
    flog.error("ERROR!", name = log.name)
    flog.error(e, name = log.name)
  })
}
